# database.py - ENHANCED VERSION WITH BETTER ERROR HANDLING
import mysql.connector
from mysql.connector import Error
import pandas as pd
from datetime import datetime
import os

class StressDatabase:
    def __init__(self, host='localhost', user='root', password='', database='plant_stress_db'):
        self.host = host
        self.user = user
        self.password = password
        self.database = database
        self.connection = None
        print(f"Database config - Host: {host}, User: {user}, DB: {database}")
    
    def connect(self):
        """Connect to MySQL database with enhanced error handling"""
        try:
            print("Attempting to connect to MySQL database...")
            
            self.connection = mysql.connector.connect(
                host=self.host,
                user=self.user,
                password=self.password,
                database=self.database,
                auth_plugin='mysql_native_password'
            )
            
            if self.connection.is_connected():
                db_info = self.connection.get_server_info()
                print(f"✅ SUCCESS: Connected to MySQL Server version {db_info}")
                return True
                
        except Error as e:
            print(f"❌ Connection failed: {e}")
            print("Attempting to create database...")
            return self.create_database()
    
    def create_database(self):
        """Create database and tables if they don't exist"""
        try:
            print("Creating database and tables...")
            
            # First connect without database to create it
            temp_conn = mysql.connector.connect(
                host=self.host,
                user=self.user,
                password=self.password,
                auth_plugin='mysql_native_password'
            )
            cursor = temp_conn.cursor()
            
            # Create database if not exists
            cursor.execute(f"CREATE DATABASE IF NOT EXISTS {self.database}")
            print(f"✅ Database '{self.database}' created successfully!")
            
            cursor.close()
            temp_conn.close()
            
            # Now connect to the specific database
            self.connection = mysql.connector.connect(
                host=self.host,
                user=self.user,
                password=self.password,
                database=self.database,
                auth_plugin='mysql_native_password'
            )
            
            # Create tables
            self.create_tables()
            print("✅ Tables created successfully!")
            return True
            
        except Error as e:
            print(f"❌ Database creation error: {e}")
            print("\nTROUBLESHOOTING TIPS:")
            print("1. Check if MySQL server is running")
            print("2. Verify your MySQL username and password")
            print("3. Try common passwords: '', 'root', 'password', '123456'")
            print("4. Make sure MySQL service is started")
            return False

    def create_tables(self):
        """Create all required tables"""
        tables = [
            """
            CREATE TABLE IF NOT EXISTS experiments (
                id INT AUTO_INCREMENT PRIMARY KEY,
                experiment_code VARCHAR(50) UNIQUE NOT NULL,
                experiment_name VARCHAR(255) NOT NULL,
                plant_species VARCHAR(100) NOT NULL,
                stress_type ENUM('drought', 'salt', 'heat', 'cold', 'flooding', 
                               'nutrient', 'UV', 'biotic', 'combined', 'heavy_metal') NOT NULL,
                researcher VARCHAR(100),
                start_date DATE,
                end_date DATE,
                description TEXT,
                status ENUM('active', 'completed', 'cancelled') DEFAULT 'active',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
            )
            """,
            """
            CREATE TABLE IF NOT EXISTS treatments (
                id INT AUTO_INCREMENT PRIMARY KEY,
                experiment_id INT NOT NULL,
                treatment_name VARCHAR(100) NOT NULL,
                treatment_type ENUM('control', 'stress', 'recovery') DEFAULT 'stress',
                stress_level DECIMAL(5,2) DEFAULT 0.0,
                description TEXT,
                replication_count INT DEFAULT 3,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (experiment_id) REFERENCES experiments(id) ON DELETE CASCADE,
                UNIQUE KEY unique_treatment (experiment_id, treatment_name)
            )
            """,
            """
            CREATE TABLE IF NOT EXISTS measurements (
                id INT AUTO_INCREMENT PRIMARY KEY,
                treatment_id INT NOT NULL,
                measurement_date DATE NOT NULL,
                plant_height DECIMAL(8,2),
                leaf_area DECIMAL(8,2),
                biomass DECIMAL(8,2),
                chlorophyll_content DECIMAL(8,2),
                relative_water_content DECIMAL(5,2),
                photosynthesis_rate DECIMAL(8,2),
                stomatal_conductance DECIMAL(8,4),
                antioxidant_activity DECIMAL(8,2),
                proline_content DECIMAL(8,2),
                notes TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (treatment_id) REFERENCES treatments(id) ON DELETE CASCADE
            )
            """
        ]
        
        cursor = self.connection.cursor()
        for i, table in enumerate(tables, 1):
            try:
                cursor.execute(table)
                print(f"✅ Table {i} created successfully")
            except Error as e:
                print(f"❌ Error creating table {i}: {e}")
        
        self.connection.commit()
        cursor.close()
    
    def execute_query(self, query, params=None):
        """Execute SQL query with error handling"""
        try:
            if not self.connection or not self.connection.is_connected():
                if not self.connect():
                    return None
            
            cursor = self.connection.cursor()
            cursor.execute(query, params or ())
            
            if query.strip().upper().startswith('SELECT'):
                results = cursor.fetchall()
                cursor.close()
                return results
            else:
                self.connection.commit()
                cursor.close()
                return True
                
        except Error as e:
            print(f"❌ Query execution error: {e}")
            print(f"Failed query: {query}")
            return None

    def test_connection(self):
        """Test database connection and return status"""
        try:
            if self.connect():
                cursor = self.connection.cursor()
                cursor.execute("SELECT VERSION()")
                version = cursor.fetchone()
                cursor.close()
                return f"✅ Connected to MySQL {version[0]}"
            else:
                return "❌ Failed to connect to database"
        except Error as e:
            return f"❌ Connection test failed: {e}"

    def export_to_excel(self, filename='plant_stress_data.xlsx'):
        """Export all data to Excel file"""
        try:
            with pd.ExcelWriter(filename, engine='openpyxl') as writer:
                # Export experiments
                experiments = self.execute_query("SELECT * FROM experiments")
                if experiments:
                    exp_df = pd.DataFrame(experiments, columns=[
                        'ID', 'Code', 'Name', 'Species', 'Stress Type', 'Researcher',
                        'Start Date', 'End Date', 'Description', 'Status', 'Created', 'Updated'
                    ])
                    exp_df.to_excel(writer, sheet_name='Experiments', index=False)
                    print(f"✅ Exported {len(exp_df)} experiments")
                
                # Export treatments
                treatments = self.execute_query("""
                    SELECT t.*, e.experiment_code 
                    FROM treatments t 
                    JOIN experiments e ON t.experiment_id = e.id
                """)
                if treatments:
                    treat_df = pd.DataFrame(treatments, columns=[
                        'ID', 'Experiment ID', 'Treatment Name', 'Type', 'Stress Level',
                        'Description', 'Replications', 'Created', 'Experiment Code'
                    ])
                    treat_df.to_excel(writer, sheet_name='Treatments', index=False)
                    print(f"✅ Exported {len(treat_df)} treatments")
                
                # Export measurements
                measurements = self.execute_query("""
                    SELECT m.*, t.treatment_name, e.experiment_code
                    FROM measurements m
                    JOIN treatments t ON m.treatment_id = t.id
                    JOIN experiments e ON t.experiment_id = e.id
                """)
                if measurements:
                    meas_df = pd.DataFrame(measurements, columns=[
                        'ID', 'Treatment ID', 'Date', 'Height', 'Leaf Area', 'Biomass',
                        'Chlorophyll', 'Water Content', 'Photosynthesis', 'Stomatal Conductance',
                        'Antioxidant', 'Proline', 'Notes', 'Created', 'Treatment Name', 'Experiment Code'
                    ])
                    meas_df.to_excel(writer, sheet_name='Measurements', index=False)
                    print(f"✅ Exported {len(meas_df)} measurements")
            
            print(f"✅ Data exported successfully to {filename}")
            return True
            
        except Exception as e:
            print(f"❌ Export error: {e}")
            return False
    
    def close(self):
        """Close database connection"""
        if self.connection and self.connection.is_connected():
            self.connection.close()
            print("Database connection closed")